<?php
/**
* 2007-2020 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author    PrestaShop SA <contact@prestashop.com>
*  @copyright 2007-2020 PrestaShop SA
*  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

if (!defined('_PS_VERSION_')) {
    exit;
}

$CLIENT_SCHEMA = stripos($_SERVER['SERVER_PROTOCOL'],'https') === true ? 'https://' : 'http://';
define("OTP_EXCANGE_RATE_LINK","{$CLIENT_SCHEMA}cdn.payments.holest.com/exchangerate.php?from=%s&to=%s");

class Otppresta extends PaymentModule
{
    protected $config_form = false;
	
	function loadCurrencies(){
		global $otp_currencies;
		
		if(isset($otp_currencies))
			return $otp_currencies;
		
		$otp_currencies = json_decode('{"AED":{"currency_name":"United Arab Emirates dirham","currency_code_3":"AED","currency_numeric_code":"784"},"ALL":{"currency_name":"Albanian lek","currency_code_3":"ALL","currency_numeric_code":"8"},"ANG":{"currency_name":"Netherlands Antillean gulden","currency_code_3":"ANG","currency_numeric_code":"532"},"ARS":{"currency_name":"Argentine peso","currency_code_3":"ARS","currency_numeric_code":"32"},"AUD":{"currency_name":"Australian dollar","currency_code_3":"AUD","currency_numeric_code":"36"},"AWG":{"currency_name":"Aruban florin","currency_code_3":"AWG","currency_numeric_code":"533"},"BBD":{"currency_name":"Barbadian dollar","currency_code_3":"BBD","currency_numeric_code":"52"},"BDT":{"currency_name":"Bangladeshi taka","currency_code_3":"BDT","currency_numeric_code":"50"},"BHD":{"currency_name":"Bahraini dinar","currency_code_3":"BHD","currency_numeric_code":"48"},"BIF":{"currency_name":"Burundian franc","currency_code_3":"BIF","currency_numeric_code":"108"},"BND":{"currency_name":"Brunei dollar","currency_code_3":"BND","currency_numeric_code":"96"},"BOB":{"currency_name":"Bolivian boliviano","currency_code_3":"BOB","currency_numeric_code":"68"},"BRL":{"currency_name":"Brazilian real","currency_code_3":"BRL","currency_numeric_code":"986"},"BSD":{"currency_name":"Bahamian dollar","currency_code_3":"BSD","currency_numeric_code":"44"},"BWP":{"currency_name":"Botswana pula","currency_code_3":"BWP","currency_numeric_code":"72"},"BZD":{"currency_name":"Belize dollar","currency_code_3":"BZD","currency_numeric_code":"84"},"CAD":{"currency_name":"Canadian dollar","currency_code_3":"CAD","currency_numeric_code":"124"},"CHF":{"currency_name":"Swiss franc","currency_code_3":"CHF","currency_numeric_code":"756"},"CLP":{"currency_name":"Chilean peso","currency_code_3":"CLP","currency_numeric_code":"152"},"CNY":{"currency_name":"Chinese renminbi yuan","currency_code_3":"CNY","currency_numeric_code":"156"},"COP":{"currency_name":"Colombian peso","currency_code_3":"COP","currency_numeric_code":"170"},"CRC":{"currency_name":"Costa Rican col\u00f3n","currency_code_3":"CRC","currency_numeric_code":"188"},"CZK":{"currency_name":"Czech koruna","currency_code_3":"CZK","currency_numeric_code":"203"},"CUP":{"currency_name":"Cuban peso","currency_code_3":"CUP","currency_numeric_code":"192"},"CVE":{"currency_name":"Cape Verdean escudo","currency_code_3":"CVE","currency_numeric_code":"132"},"DKK":{"currency_name":"Danish krone","currency_code_3":"DKK","currency_numeric_code":"208"},"DOP":{"currency_name":"Dominican peso","currency_code_3":"DOP","currency_numeric_code":"214"},"DZD":{"currency_name":"Algerian dinar","currency_code_3":"DZD","currency_numeric_code":"12"},"EGP":{"currency_name":"Egyptian pound","currency_code_3":"EGP","currency_numeric_code":"818"},"ETB":{"currency_name":"Ethiopian birr","currency_code_3":"ETB","currency_numeric_code":"230"},"EUR":{"currency_name":"Euro","currency_code_3":"EUR","currency_numeric_code":"978"},"FJD":{"currency_name":"Fijian dollar","currency_code_3":"FJD","currency_numeric_code":"242"},"GBP":{"currency_name":"British pound","currency_code_3":"GBP","currency_numeric_code":"826"},"GIP":{"currency_name":"Gibraltar pound","currency_code_3":"GIP","currency_numeric_code":"292"},"GMD":{"currency_name":"Gambian dalasi","currency_code_3":"GMD","currency_numeric_code":"270"},"GNF":{"currency_name":"Guinean franc","currency_code_3":"GNF","currency_numeric_code":"324"},"GTQ":{"currency_name":"Guatemalan quetzal","currency_code_3":"GTQ","currency_numeric_code":"320"},"GYD":{"currency_name":"Guyanese dollar","currency_code_3":"GYD","currency_numeric_code":"328"},"HKD":{"currency_name":"Hong Kong dollar","currency_code_3":"HKD","currency_numeric_code":"344"},"HNL":{"currency_name":"Honduran lempira","currency_code_3":"HNL","currency_numeric_code":"340"},"HTG":{"currency_name":"Haitian gourde","currency_code_3":"HTG","currency_numeric_code":"332"},"HUF":{"currency_name":"Hungarian forint","currency_code_3":"HUF","currency_numeric_code":"348"},"IDR":{"currency_name":"Indonesian rupiah","currency_code_3":"IDR","currency_numeric_code":"360"},"ILS":{"currency_name":"Israeli new sheqel","currency_code_3":"ILS","currency_numeric_code":"376"},"INR":{"currency_name":"Indian rupee","currency_code_3":"INR","currency_numeric_code":"356"},"IQD":{"currency_name":"Iraqi dinar","currency_code_3":"IQD","currency_numeric_code":"368"},"IRR":{"currency_name":"Iranian rial","currency_code_3":"IRR","currency_numeric_code":"364"},"JMD":{"currency_name":"Jamaican dollar","currency_code_3":"JMD","currency_numeric_code":"388"},"JOD":{"currency_name":"Jordanian dinar","currency_code_3":"JOD","currency_numeric_code":"400"},"JPY":{"currency_name":"Japanese yen","currency_code_3":"JPY","currency_numeric_code":"392"},"KES":{"currency_name":"Kenyan shilling","currency_code_3":"KES","currency_numeric_code":"404"},"KHR":{"currency_name":"Cambodian riel","currency_code_3":"KHR","currency_numeric_code":"116"},"KMF":{"currency_name":"Comorian franc","currency_code_3":"KMF","currency_numeric_code":"174"},"KRW":{"currency_name":"South Korean won","currency_code_3":"KRW","currency_numeric_code":"410"},"KWD":{"currency_name":"Kuwaiti dinar","currency_code_3":"KWD","currency_numeric_code":"414"},"LAK":{"currency_name":"Lao kip","currency_code_3":"LAK","currency_numeric_code":"418"},"LBP":{"currency_name":"Lebanese pound","currency_code_3":"LBP","currency_numeric_code":"422"},"LKR":{"currency_name":"Sri Lankan rupee","currency_code_3":"LKR","currency_numeric_code":"144"},"LRD":{"currency_name":"Liberian dollar","currency_code_3":"LRD","currency_numeric_code":"430"},"LSL":{"currency_name":"Lesotho loti","currency_code_3":"LSL","currency_numeric_code":"426"},"LYD":{"currency_name":"Libyan dinar","currency_code_3":"LYD","currency_numeric_code":"434"},"MAD":{"currency_name":"Moroccan dirham","currency_code_3":"MAD","currency_numeric_code":"504"},"MNT":{"currency_name":"Mongolian t\u00f6gr\u00f6g","currency_code_3":"MNT","currency_numeric_code":"496"},"MOP":{"currency_name":"Macanese pataca","currency_code_3":"MOP","currency_numeric_code":"446"},"MUR":{"currency_name":"Mauritian rupee","currency_code_3":"MUR","currency_numeric_code":"480"},"MVR":{"currency_name":"Maldivian rufiyaa","currency_code_3":"MVR","currency_numeric_code":"462"},"MWK":{"currency_name":"Malawian kwacha","currency_code_3":"MWK","currency_numeric_code":"454"},"MYR":{"currency_name":"Malaysian ringgit","currency_code_3":"MYR","currency_numeric_code":"458"},"NGN":{"currency_name":"Nigerian naira","currency_code_3":"NGN","currency_numeric_code":"566"},"NOK":{"currency_name":"Norwegian krone","currency_code_3":"NOK","currency_numeric_code":"578"},"NPR":{"currency_name":"Nepalese rupee","currency_code_3":"NPR","currency_numeric_code":"524"},"NZD":{"currency_name":"New Zealand dollar","currency_code_3":"NZD","currency_numeric_code":"554"},"OMR":{"currency_name":"Omani rial","currency_code_3":"OMR","currency_numeric_code":"512"},"PAB":{"currency_name":"Panamanian balboa","currency_code_3":"PAB","currency_numeric_code":"590"},"PEN":{"currency_name":"Peruvian nuevo sol","currency_code_3":"PEN","currency_numeric_code":"604"},"PGK":{"currency_name":"Papua New Guinean kina","currency_code_3":"PGK","currency_numeric_code":"598"},"PHP":{"currency_name":"Philippine peso","currency_code_3":"PHP","currency_numeric_code":"608"},"PKR":{"currency_name":"Pakistani rupee","currency_code_3":"PKR","currency_numeric_code":"586"},"PLN":{"currency_name":"Polish Z\u0142oty","currency_code_3":"PLN","currency_numeric_code":"985"},"PYG":{"currency_name":"Paraguayan guaran\u00ed","currency_code_3":"PYG","currency_numeric_code":"600"},"QAR":{"currency_name":"Qatari riyal","currency_code_3":"QAR","currency_numeric_code":"634"},"RON":{"currency_name":"Romanian leu","currency_code_3":"RON","currency_numeric_code":"946"},"RWF":{"currency_name":"Rwandan franc","currency_code_3":"RWF","currency_numeric_code":"646"},"SAR":{"currency_name":"Saudi riyal","currency_code_3":"SAR","currency_numeric_code":"682"},"SBD":{"currency_name":"Solomon Islands dollar","currency_code_3":"SBD","currency_numeric_code":"90"},"SCR":{"currency_name":"Seychellois rupee","currency_code_3":"SCR","currency_numeric_code":"690"},"SEK":{"currency_name":"Swedish krona","currency_code_3":"SEK","currency_numeric_code":"752"},"SGD":{"currency_name":"Singapore dollar","currency_code_3":"SGD","currency_numeric_code":"702"},"SLL":{"currency_name":"Sierra Leonean leone","currency_code_3":"SLL","currency_numeric_code":"694"},"SOS":{"currency_name":"Somali shilling","currency_code_3":"SOS","currency_numeric_code":"706"},"RUB":{"currency_name":"Russian ruble","currency_code_3":"RUB","currency_numeric_code":"643"},"SVC":{"currency_name":"Salvadoran col\u00f3n","currency_code_3":"SVC","currency_numeric_code":"222"},"SYP":{"currency_name":"Syrian pound","currency_code_3":"SYP","currency_numeric_code":"760"},"SZL":{"currency_name":"Swazi lilangeni","currency_code_3":"SZL","currency_numeric_code":"748"},"THB":{"currency_name":"Thai baht","currency_code_3":"THB","currency_numeric_code":"764"},"TND":{"currency_name":"Tunisian dinar","currency_code_3":"TND","currency_numeric_code":"788"},"TOP":{"currency_name":"Tongan pa\u02bbanga","currency_code_3":"TOP","currency_numeric_code":"776"},"TRY":{"currency_name":"Turkish new lira","currency_code_3":"TRY","currency_numeric_code":"949"},"TTD":{"currency_name":"Trinidad and Tobago dollar","currency_code_3":"TTD","currency_numeric_code":"780"},"TWD":{"currency_name":"New Taiwan dollar","currency_code_3":"TWD","currency_numeric_code":"901"},"TZS":{"currency_name":"Tanzanian shilling","currency_code_3":"TZS","currency_numeric_code":"834"},"USD":{"currency_name":"United States dollar","currency_code_3":"USD","currency_numeric_code":"840"},"VND":{"currency_name":"Vietnamese Dong","currency_code_3":"VND","currency_numeric_code":"704"},"VUV":{"currency_name":"Vanuatu vatu","currency_code_3":"VUV","currency_numeric_code":"548"},"WST":{"currency_name":"Samoan tala","currency_code_3":"WST","currency_numeric_code":"882"},"YER":{"currency_name":"Yemeni rial","currency_code_3":"YER","currency_numeric_code":"886"},"RSD":{"currency_name":"Serbian dinar","currency_code_3":"RSD","currency_numeric_code":"941"},"ZAR":{"currency_name":"South African rand","currency_code_3":"ZAR","currency_numeric_code":"710"},"AMD":{"currency_name":"Armenian dram","currency_code_3":"AMD","currency_numeric_code":"51"},"MMK":{"currency_name":"Myanmar kyat","currency_code_3":"MMK","currency_numeric_code":"104"},"HRK":{"currency_name":"Croatian kuna","currency_code_3":"HRK","currency_numeric_code":"191"},"ERN":{"currency_name":"Eritrean nakfa","currency_code_3":"ERN","currency_numeric_code":"232"},"DJF":{"currency_name":"Djiboutian franc","currency_code_3":"DJF","currency_numeric_code":"262"},"ISK":{"currency_name":"Icelandic kr\u00f3na","currency_code_3":"ISK","currency_numeric_code":"352"},"KZT":{"currency_name":"Kazakhstani tenge","currency_code_3":"KZT","currency_numeric_code":"398"},"KGS":{"currency_name":"Kyrgyzstani som","currency_code_3":"KGS","currency_numeric_code":"417"},"MXN":{"currency_name":"Mexican peso","currency_code_3":"MXN","currency_numeric_code":"484"},"MDL":{"currency_name":"Moldovan leu","currency_code_3":"MDL","currency_numeric_code":"498"},"NAD":{"currency_name":"Namibian dollar","currency_code_3":"NAD","currency_numeric_code":"516"},"NIO":{"currency_name":"Nicaraguan c\u00f3rdoba","currency_code_3":"NIO","currency_numeric_code":"558"},"UGX":{"currency_name":"Ugandan shilling","currency_code_3":"UGX","currency_numeric_code":"800"},"MKD":{"currency_name":"Macedonian denar","currency_code_3":"MKD","currency_numeric_code":"807"},"UYU":{"currency_name":"Uruguayan peso","currency_code_3":"UYU","currency_numeric_code":"858"},"UZS":{"currency_name":"Uzbekistani som","currency_code_3":"UZS","currency_numeric_code":"860"},"AZN":{"currency_name":"Azerbaijani manat","currency_code_3":"AZN","currency_numeric_code":"934"},"GHS":{"currency_name":"Ghanaian cedi","currency_code_3":"GHS","currency_numeric_code":"936"},"SDG":{"currency_name":"Sudanese pound","currency_code_3":"SDG","currency_numeric_code":"938"},"MZN":{"currency_name":"Mozambican metical","currency_code_3":"MZN","currency_numeric_code":"943"},"XAF":{"currency_name":"Central African CFA franc","currency_code_3":"XAF","currency_numeric_code":"950"},"XCD":{"currency_name":"East Caribbean dollar","currency_code_3":"XCD","currency_numeric_code":"951"},"XOF":{"currency_name":"West African CFA franc","currency_code_3":"XOF","currency_numeric_code":"952"},"XPF":{"currency_name":"CFP franc","currency_code_3":"XPF","currency_numeric_code":"953"},"SRD":{"currency_name":"Surinamese dollar","currency_code_3":"SRD","currency_numeric_code":"968"},"MGA":{"currency_name":"Malagasy ariary","currency_code_3":"MGA","currency_numeric_code":"969"},"AFN":{"currency_name":"Afghan afghani","currency_code_3":"AFN","currency_numeric_code":"971"},"TJS":{"currency_name":"Tajikistani somoni","currency_code_3":"TJS","currency_numeric_code":"972"},"AOA":{"currency_name":"Angolan kwanza","currency_code_3":"AOA","currency_numeric_code":"973"},"BGN":{"currency_name":"Bulgarian lev","currency_code_3":"BGN","currency_numeric_code":"975"},"CDF":{"currency_name":"Congolese franc","currency_code_3":"CDF","currency_numeric_code":"976"},"BAM":{"currency_name":"Bosnia and Herzegovina convert","currency_code_3":"BAM","currency_numeric_code":"977"},"UAH":{"currency_name":"Ukrainian hryvnia","currency_code_3":"UAH","currency_numeric_code":"980"},"GEL":{"currency_name":"Georgian lari","currency_code_3":"GEL","currency_numeric_code":"981"}}', true);	
		
		return $otp_currencies;
	}

    public function __construct()
    {
		
		
        $this->name = 'otppresta';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.0';
        $this->author = 'Holest Engineering';
        $this->need_instance = 1;

        /**
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;
		parent::__construct();
		
		$this->displayName = $this->l('OTP payments for PrestaShop','otppresta');
        
		$this->description = $this->l('Adds OTP as payment option to your PrestaShop','otppresta');
		
		require(__DIR__ . DIRECTORY_SEPARATOR . "settings.php");
		
		$this->is_eu_compatible = 1;
		
		$this->payment_currency = "RSD";

        $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);
		
		$this->active = Configuration::get("otppresta_enabled","") == "yes";
		
		$this->module_base_url = $this->_path;
		

		
	}
	
	public function normalText($str){
		return str_ireplace(array('u0161','u0111','u010D','u0107','u0160','u0110','u010C','u0106','u017e','u017d','u2013'),array('š','đ','č','ć','Š','Đ','Č','Ć','ž','Ž','-'),$str);
	}
	
	function execute_curl_request($url, $data){
		$ch = curl_init($url);
		$payload = json_encode($data, JSON_NUMERIC_CHECK);
		
		curl_setopt($ch, CURLOPT_URL, $url );
		curl_setopt($ch, CURLOPT_POST, true);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
		curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type:application/json'));
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
		curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

		$result = curl_exec($ch);
		
		$response = new stdClass;
		$response->raw     = $result;
		$response->request = $data;
		
		if($errno = curl_errno($ch)) {
			$error_message = curl_strerror($errno);
			$response->errno = $errno;
			$response->error_message = $error_message;
		}
		
		$response->response = json_decode($result,true);
		if($response->response){
			foreach($response->response as $key => $val){
				$response->response[$key] = $this->normalText($val);	
			}
		}
		
		if(!isset($response->response)){
			$response->errno         = 400;
			$response->error_message = "Missing response from service";
		}else if(isset($response->response["error"])){
			$response->errno = $response->response["error"]["code"];
			$response->error_message = $response->response["error"]["message"];
		} 
		curl_close($ch);
		return $response;
	}
	
	public function curl_get($url){
		$ch  = curl_init () or exit ( "curl error: Can't init curl" );
		$url = trim ( $url );
		
		curl_setopt ( $ch, CURLOPT_URL, $url );
		curl_setopt ( $ch, CURLOPT_POST, 0 );
		curl_setopt ( $ch, CURLOPT_CUSTOMREQUEST, "GET");
		curl_setopt ( $ch, CURLOPT_RETURNTRANSFER, true );
		curl_setopt ( $ch, CURLOPT_CONNECTTIMEOUT, 30 );
		curl_setopt ( $ch, CURLOPT_TIMEOUT, 60 );
		curl_setopt ( $ch, CURLOPT_USERAGENT, "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/48.0.2564.109 Safari/537.36");
		curl_setopt ( $ch, CURLOPT_FOLLOWLOCATION, TRUE);

		$response = curl_exec ( $ch );
		
		if (!$response) {
			$err = "Curl errno: " . curl_errno ( $ch ) . " (" . $url . " )\n";
			$err .= "Curl error: " . curl_error ( $ch ) . "\n";
			$info = curl_getinfo($ch);
			$err .= "HTTP code: ".$info["http_code"]."\n";
			throw new Exception($err);
		}
		
		curl_close ( $ch );
		return $this->normalText($response);
	}
	
	public function curl_post($url, $postDataArr){
		$ch  = curl_init () or exit ( "curl error: Can't init curl" );
		$url = trim ( $url );
		
		curl_setopt ( $ch, CURLOPT_URL, $url );
		curl_setopt ( $ch, CURLOPT_POST, 1 );
		curl_setopt ( $ch, CURLOPT_CUSTOMREQUEST, "POST");
		curl_setopt ( $ch, CURLOPT_POSTFIELDS, http_build_query($postDataArr));
		curl_setopt ( $ch, CURLOPT_RETURNTRANSFER, true );
		curl_setopt ( $ch, CURLOPT_CONNECTTIMEOUT, 30 );
		curl_setopt ( $ch, CURLOPT_TIMEOUT, 60 );
		curl_setopt ( $ch, CURLOPT_USERAGENT, "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/48.0.2564.109 Safari/537.36");
		curl_setopt ( $ch, CURLOPT_FOLLOWLOCATION, TRUE);

		$response = curl_exec ( $ch );
		
		if (!$response) {
			$err = "Curl errno: " . curl_errno ( $ch ) . " (" . $url . " )\n";
			$err .= "Curl error: " . curl_error ( $ch ) . "\n";
			$info = curl_getinfo($ch);
			$err .= "HTTP code: ".$info["http_code"]."\n";
			throw new Exception($err);
		}
		
		curl_close ( $ch );
		return $this->normalText($response);
	}
	
	public function getExchangeRate($from_currency, $to_currency){
		$exchange_rate = null;
		if($from_currency == $to_currency){
			$exchange_rate = 1.0;
		}else{
			$fromto        = strtoupper($from_currency .'_'. $to_currency);
			$exchange_resp = json_decode(Configuration::get("otppresta_excange_{$fromto}",null));
			if($exchange_resp){
				if(isset($exchange_resp->ts)){
					if($exchange_resp->ts + 60 * 30 < time()){
						if(isset($exchange_resp->rate)){
							$exchange_rate = floatval($exchange_resp->rate);	
						}
					}
				}
			}
			
			if($exchange_rate === null){
				$excresp = null;
				
				try{
					$excresp = $this->curl_get(sprintf(OTP_EXCANGE_RATE_LINK,strtoupper($from_currency),strtoupper($to_currency)));
					$excresp = json_decode($excresp);
				}catch(Exception $ex){
					$excresp = null;
				}
				
				if($excresp){
					$excresp->ts = time();
					Configuration::updateValue("otppresta_excange_{$fromto}", json_encode($excresp));
					$exchange_rate  = floatval($excresp->rate); 
				}
				
				if(!$excresp && $exchange_resp){
					if(isset($exchange_resp->rate)){
						$exchange_rate = floatval($exchange_resp->rate);
					}
				}
			}
			
			if(!$exchange_rate)
				$exchange_rate = 1.0;
			
			if(floatval(Configuration::get("otppresta_conversion_rate_adjust",0))){
				$exchange_rate = $exchange_rate * (1.0 + floatval(Configuration::get("otppresta_conversion_rate_adjust",0)) / 100);
			}
		}
		return $exchange_rate;
	}
	
	function strToHex($string){
		$hex='';
		for ($i=0; $i < strlen($string); $i++){
			$hex .= dechex(ord($string[$i]));
		}
		return $hex;
	}
	
	function generateHash( $storename, $txndatetime, $chargetotal, $currency_num_code, $sharedsecret) {
		$asciihex = $this->strToHex("{$storename}{$txndatetime}{$chargetotal}{$currency_num_code}{$sharedsecret}");
		return hash("sha256",$asciihex);
	}
	
	public function storeTransactionData($order_id, $type, $data){
		try{
			$store_data = $data;
			
			if(!is_string($store_data))
				$store_data = json_encode($store_data);
			
			$existing = Db::getInstance()->ExecuteS("SELECT * FROM " . _DB_PREFIX_ . "otp_transactions WHERE uid = '{$order_id}_{$type}';");
			if(!empty($existing)){
				Db::getInstance()->Execute("DELETE FROM " . _DB_PREFIX_ . "otp_transactions WHERE uid = '{$order_id}_{$type}';");
			}
			
			return Db::getInstance()->insert('otp_transactions', array(
				'uid'       => pSQL("{$order_id}_{$type}"),
				'data'      => pSQL($store_data)
			));
		}catch(Exception $ex){
			return null;
		}
	}
	
	public function getTransactionData($order_id, $type){
		try{
			$existing = Db::getInstance()->ExecuteS("SELECT * FROM " . _DB_PREFIX_ . "otp_transactions WHERE uid = '{$order_id}_{$type}';");
			if(!empty($existing)){
				return json_decode($existing[0]["data"],true);
			}
			return null;
		}catch(Exception $ex){
			return null;
		}
	}
	
	public function getOrderTransactions($order_id){
		$transactions = array();
		
		try{
			$existing = Db::getInstance()->ExecuteS("SELECT * FROM " . _DB_PREFIX_ . "otp_transactions WHERE uid LIKE '{$order_id}_%';");
			if(!empty($existing)){
				foreach($existing as $rec){
					$t =  json_decode($rec["data"],true);	
					$uid = explode("_",$rec["uid"]);
					$transactions[$uid[1]] = $t;
				}
			}
			return $transactions;
		}catch(Exception $ex){
			return $transactions;
		}
	}
	
	public function notifyCustomer($email, $subject, $body){
		try{
			return Mail::Send(
				(int)(Configuration::get('PS_LANG_DEFAULT')), // defaut language id
				'contact', // email template file to be use
				$subject, // email subject
				array(
					'{email}'   => Configuration::get('PS_SHOP_EMAIL'), // sender email address
					'{message}' => $body
				),
				$email, // receiver email address
				NULL, //receiver name
				NULL, //from email address
				NULL  //from name
			);
		}catch(Exception $ex){
			return null;
		}
	}
	
	public function filter_displayable($data, $lines = false){
		$displayable = array();
		
		foreach($data as $key => $val){
			if(
			   stripos($key,"status") !== false || stripos($key,"TransactionResult")  !== false
			   ||
			   stripos($key,"oid") !== false || stripos($key,"OrderId")  !== false
			   ||
			   stripos($key,"timezone") !== false
			   ||
			   stripos($key,"txndatetime") !== false
			   ||
			   stripos($key,"TransactionId") !== false
			   ||
			   stripos($key,"approval_code") !== false || stripos($key,"ApprovalCode")  !== false
			   ||
			   stripos($key,"chargetotal") !== false
			   ||
			   stripos($key,"currency") !== false
			   ||
			   stripos($key,"processor_response_code") !== false || stripos($key,"ProcessorResponseCode")  !== false
				){
				if(!$lines){	
					$displayable[$this->l(strtolower($key),'otppresta')] = $this->normalText($val);
				}else{
					$displayable[] = ($this->l(strtolower($key),'otppresta') . ": " . $this->normalText($val));
				}
			}
		}
		
		return $displayable;
	}

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        if (extension_loaded('curl') == false)
        {
            $this->_errors[] = $this->l('You have to enable the cURL extension on your server to install this module','otppresta');
            return false;
        }

        Configuration::updateValue('otppresta_enabled', false);

        return parent::install() &&
            $this->registerHook('header') &&
            $this->registerHook('backOfficeHeader') &&
            $this->registerHook('payment') &&
            $this->registerHook('paymentReturn') &&
            $this->registerHook('paymentOptions') &&
            $this->registerHook('actionOrderSlipAdd') &&
            $this->registerHook('actionOrderStatusPostUpdate') &&
            $this->registerHook('actionOrderStatusUpdate') &&
            $this->registerHook('actionPaymentConfirmation') &&
            $this->registerHook('displayBackOfficeHeader') &&
            $this->registerHook('displayBeforePayment') &&
            $this->registerHook('displayHeader') &&
            $this->registerHook('displayInvoice') &&
            $this->registerHook('displayOrderConfirmation') &&
            $this->registerHook('displayOrderDetail') &&
            $this->registerHook('displayPayment') &&
            $this->registerHook('displayPaymentReturn') &&
			$this->registerHook('displayAdminOrderLeft');
    }

    public function uninstall()
    {
        Configuration::deleteByName('otppresta_enabled');

        return parent::uninstall();
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        /**
         * If values have been submitted in the form, process.
         */
        if (((bool)Tools::isSubmit('submitOtpprestaModule')) == true) {
            $this->postProcess();
        }

        $this->context->smarty->assign('module_dir', $this->_path);

        $output = $this->context->smarty->fetch($this->local_path.'views/templates/admin/configure.tpl');

        return $output.$this->renderForm();
    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
		
		$sql = "CREATE TABLE IF NOT EXISTS " . _DB_PREFIX_ . "otp_transactions (
				  `uid` varchar(32) NOT NULL,
				  `data` text NOT NULL,
				  `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
				  PRIMARY KEY (`uid`)
				);";
		try{		
			Db::getInstance()->execute($sql);		
		}catch(Exception $ex){
			//
		}
		
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitOtpprestaModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(), /* Add values for your inputs */
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );
		
		return $helper->generateForm(array($this->getConfigForm()));
    }

    /**
     * Create the structure of your form.
     */
    protected function getConfigForm()
    {
        return array(
            'form' => array(
                'legend' => array(
                'title' => $this->l('Settings','otppresta'),
                'icon' => 'icon-cogs',
                ),
                'input' =>  $this->_form_fields,
                'submit' => array(
                    'title' => $this->l('Save','otppresta'),
                ),
            ),
        );
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
		$this->smarty->assign('module_dir', $this->_path);
		
		foreach($this->_form_fields as $name => $def){
			$tmp[$def["name"]] = Configuration::get($def["name"],"");
		}
        return $tmp;
    }

    /**
     * Save form data.
     */
    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();
		
		foreach (array_keys($form_values) as $key) {
			
			if($this->_form_fields[str_replace("otppresta_","",$key)]["type"] == "password"){
				if(!Tools::getValue($key))
					continue;
			}
            Configuration::updateValue($key, Tools::getValue($key));
        }
    }

    /**
    * Add the CSS & JavaScript files you want to be loaded in the BO.
    */
    public function hookBackOfficeHeader()
    {
		if (Tools::getValue('module_name') == $this->name) {
            $this->context->controller->addJS($this->_path.'views/js/back.js');
            $this->context->controller->addCSS($this->_path.'views/css/back.css');
        }
    }

    /**
     * Add the CSS & JavaScript files you want to be added on the FO.
     */
    public function hookHeader()
    {
        $this->context->controller->addJS($this->_path.'/views/js/front.js');
        $this->context->controller->addCSS($this->_path.'/views/css/front.css');
    }

    /**
     * This method is used to render the payment button,
     * Take care if the button should be displayed or not.
     */
    public function hookPayment($params)
    {
        $currency_id = $params['cart']->id_currency;
        $currency = new Currency((int)$currency_id);

        if (in_array($currency->iso_code, $this->limited_currencies) == false)
            return false;

        $this->smarty->assign('module_dir', $this->_path);

        return $this->display(__FILE__, 'views/templates/hook/payment.tpl');
    }

    /**
     * This hook is used to display the order confirmation page.
     */
    public function hookPaymentReturn($params)
    {
        if ($this->active == false)
            return;

        $order = $params['objOrder'];

        if ($order->getCurrentOrderState()->id != Configuration::get('PS_OS_ERROR'))
            $this->smarty->assign('status', 'ok');

        $this->smarty->assign(array(
            'id_order' => $order->id,
            'reference' => $order->reference,
            'params' => $params,
            'total' => Tools::displayPrice($params['total_to_pay'], $params['currencyObj'], false),
        ));

        return $this->display(__FILE__, 'views/templates/hook/confirmation.tpl');
    }
	
	public function getCustomerUid($email){
		$email = trim(strtolower($email));
		$emailmd5 =  md5($email);
		
		$customer_duid = Configuration::get("otppresta_{$emailmd5}",'');
		
		if(!$customer_duid){
			$customer_duid = uniqid($emailmd5);
			Configuration::updateValue("otppresta_{$emailmd5}", $customer_duid);
		}
		return md5($customer_duid.$emailmd5);
	}

	public function generateForm(){
		
		 $date = new DateTime("now", new DateTimeZone("Europe/Berlin"));
		 $txndatetime       = $date->format('Y:m:d-H:i:s');
		
		 $cart_id     = $this->context->cart->id;
         $customer_id = $this->context->cart->id_customer;
         $currency    = $this->context->currency;
		 
		 $oid               = $cart_id;
		 $customer_uid      = $this->getCustomerUid($this->context->customer->email);
		 $chargetotal       = (float)$this->context->cart->getOrderTotal(true, Cart::BOTH);
		 
		 $payment_currency  = Configuration::get("otppresta_payment_currency","RSD");
		 $otp_currencies   = $this->loadCurrencies();
		 
		 $currency_num_code = $otp_currencies[$payment_currency]["currency_numeric_code"];
		  
		 if($payment_currency != $currency->iso_code){
			 $ExchangeRate = $this->getExchangeRate($currency->iso_code, $payment_currency);
			 $chargetotal = $chargetotal * $ExchangeRate;
		 }
		 
		 $chargetotal     = number_format(round(floatval($chargetotal),2),2, '.', '');
		
		 $hash            = $this->generateHash( Configuration::get("otppresta_store_sid",""), 	
												 $txndatetime, 
												 $chargetotal, 
												 $currency_num_code, 
												 Configuration::get("otppresta_shared_secret",""));
												 
		 $secure_key = Context::getContext()->customer->secure_key;
		 
		
		 	
		 $this->context->smarty->assign([
            'action'            => $this->context->link->getModuleLink($this->name, 'redirect', array(), true),
			'timezone'          => "Europe/Berlin",
			'txntype'           => Configuration::get("otppresta_txntype","sale"),
			'txndatetime'       => $txndatetime,
			'hash'              => $hash,
			'storename'         => Configuration::get("otppresta_store_sid",""),
            'mode'              => "payonly",
			'chargetotal'       => $chargetotal,
			'currency_num'      => $currency_num_code,
			'oid'               => substr($secure_key,0,6) . $oid,
			'ponumber'          => $secure_key,  
			'responseSuccessURL'=> $this->context->link->getModuleLink($this->name, 'validation', array(), true),
			'responseFailURL'   => $this->context->link->getModuleLink($this->name, 'validation', array(), true),
			'allowhosteddataid' => Configuration::get("otppresta_allow_hosteddataid","no"),
			'hosteddataid'      => $customer_uid 
	    ]);
		
		return $this->context->smarty->fetch('module:otppresta/views/templates/front/form.tpl');
	}

    /**
     * Return payment options available for PS 1.7+
     *
     * @param array Hook parameters
     *
     * @return array|null
     */
    public function hookPaymentOptions($params)
    {
	    if (!$this->active) {
            return;
        }
       
		if(Configuration::get("test_ips","")){
			if(!$this->getRealIpAddr() || strpos(Configuration::get("test_ips",""), $this->getRealIpAddr()) === false)
				return array();
		}
		
	    $option = new \PrestaShop\PrestaShop\Core\Payment\PaymentOption();
         
		$option->setCallToActionText(Configuration::get("otppresta_title",$this->displayName))
				->setAdditionalInformation(Configuration::get("otppresta_disclaimer",""))
				->setLogo(Media::getMediaPath(_PS_MODULE_DIR_.$this->name.'/logo_small.png'))
				->setForm($this->generateForm());		
				
		return [
            $option
        ];
    }
	
	public function getRealIpAddr(){
		if (!empty($_SERVER['HTTP_CLIENT_IP'])){
		  $ip=$_SERVER['HTTP_CLIENT_IP'];
		}elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])){
		  $ip=$_SERVER['HTTP_X_FORWARDED_FOR'];
		}else{
		  $ip=$_SERVER['REMOTE_ADDR'];
		}
		return $ip;
	}
	
   
    public function hookActionOrderSlipAdd()
    {
        /* Place your code here. */
    }

    public function hookActionOrderStatusPostUpdate()
    {
        /* Place your code here. */
    }

    public function hookActionOrderStatusUpdate()
    {
        /* Place your code here. */
    }

    public function hookActionPaymentConfirmation()
    {
        /* Place your code here. */
    }

    public function hookDisplayBackOfficeHeader()
    {
		if (isset($_REQUEST["vieworder"])) {
            $this->context->controller->addJS($this->_path.'views/js/backorder.js');
        }
		
		if(isset($_POST["otp_do_transaction"])){
			$order_id = (int)$_POST["otp_transaction_order_id"];
			$transactions = $this->getOrderTransactions($order_id);
			
			if($_POST["otp_do_transaction"] == "void"){
				
				$on_transaction = null;
				$is_cof         = false;
				
				if(isset($transactions["sale"])){
					if(isset($transactions["sale"]["processor_response_code"])){
						if($transactions["sale"]["processor_response_code"] == "00"){
							$on_transaction = $transactions["sale"];
						}
					}
					
					if(isset($transactions["sale"]["ProcessorResponseCode"])){
						if($transactions["sale"]["ProcessorResponseCode"] == "00"){
							$on_transaction = $transactions["sale"];
							$is_cof = true;
						}
					}
				}
				
				if(!$on_transaction){
					if(isset($transactions["postauth"])){
						if(isset($transactions["postauth"]["processor_response_code"])){
							if($transactions["postauth"]["processor_response_code"] == "00"){
								$on_transaction = $transactions["postauth"];
							}
						}
					}
				}
				
				if(!$on_transaction){
					if(isset($transactions["preauth"])){
						if(isset($transactions["preauth"]["processor_response_code"])){
							if($transactions["preauth"]["processor_response_code"] == "00"){
								$on_transaction = $transactions["preauth"];
							}
						}
					}
				}
				
				if($on_transaction){
					if($is_cof)
						$this->executeCofVoid($order_id, $on_transaction);
					else
						$this->executeVoid($order_id, $on_transaction);
				}else
					$this->otp_error_message = $this->l('No referent transaction for VOID','otppresta');
					
				
				
			}else if($_POST["otp_do_transaction"] == "capture"){
				
				$amt                = (float)$_POST["otp_capture_amount"];
				$on_transaction = null;
				
				if(isset($transactions["preauth"])){
					if(isset($transactions["preauth"]["processor_response_code"])){
						if($transactions["preauth"]["processor_response_code"] == "00"){
							$on_transaction = $transactions["preauth"];
						}
					}
				}
				
				if($on_transaction){
					$this->executeCapture($order_id, $on_transaction, $amt);
				}else
					$this->otp_error_message = $this->l('No referent transaction for Capture/Post-Autorization','otppresta');
				
			}else if($_POST["otp_do_transaction"] == "charge"){
				
				$amt      = (float)$_POST["otp_charge_amount"];
				$otp_exchange_rate = (float)$_POST["otp_exchange_rate"];
				
				$this->executeCharge($order_id, $amt, $otp_exchange_rate);
			}
		}
    }
	
	public function executeVoid($order_id, $ref_transaction){
		$existing = $this->getTransactionData($order_id,"void");
		if($existing){
			if(isset($existing["processor_response_code"])){
				if($existing["processor_response_code"] == "00"){
					$this->otp_error_message = $this->l('VOID transaction already completed!','otppresta');
					return;
				}
			}
			if(isset($existing["ProcessorResponseCode"])){
				if($existing["ProcessorResponseCode"] == "00"){
					$this->otp_error_message = $this->l('VOID transaction already completed!','otppresta');
					return;
				}
			}
		}
			
		$date = new DateTime("now", new DateTimeZone("Europe/Berlin"));
		$txndatetime     = $date->format('Y:m:d-H:i:s');
		
		
		$chargetotal = str_replace(",",".",$ref_transaction["chargetotal"]);
		$chargetotal = number_format( floatval($chargetotal) , 2, '.', '');
		
		$hash = $this->generateHash(	Configuration::get("otppresta_store_sid",""), 
										$txndatetime, 
										$chargetotal, 
										$ref_transaction["currency"] , 
										Configuration::get("otppresta_shared_secret",""));
		
		
		$form_data = array(
			"txntype"     		 => "void",
			"timezone"    		 => "Europe/Berlin",
			"txndatetime" 		 => $txndatetime,
			"storename"   		 => Configuration::get("otppresta_store_sid",""),
			"currency"    		 => $ref_transaction["currency"],
			"oid"         		 => $ref_transaction["oid"],
			"chargetotal" 		 => $chargetotal,
			"responseSuccessURL" => $this->context->link->getModuleLink($this->name, 'readjson', array(), true),
			"responseFailURL"    => $this->context->link->getModuleLink($this->name, 'readjson', array(), true),
			"authenticateTransaction" => "false",
			"tdate"              => $ref_transaction["tdate"],
			"hash_algorithm"     => "SHA256",
			"hash"               => $hash
		);
		
		try{
			$resp = $this->curl_post(Configuration::get("otppresta_form_url"), $form_data);
			$response_data = array();
			$output_array = array();
			preg_match_all('/<input.*[value=|name=]\"(.*?)\".*[value=|name=]\"(.*?)\">/', $resp, $output_array);
			$names_index  = 2;
			$values_index = 1;
			
			if(in_array("response_hash",$output_array[1])){
				$names_index  = 1;
				$values_index = 2;
			}
			
			for($i = 0; $i < count($output_array[1]); $i++){
				$response_data[$output_array[$names_index][$i]] = $output_array[$values_index][$i];
			}
			
			if(isset($response_data["processor_response_code"])){
				$response_data["txntype"] = "void";
				$this->storeTransactionData($order_id,"void",$response_data);
				if($response_data["processor_response_code"] == "00"){
					$history = new OrderHistory();
					$history->id_order = $order_id;
					$history->changeIdOrderState((int)Configuration::get("otppresta_void_completed"), (int)$order_id);
					
					$this->otp_message = $this->l('VOID completed successfully','otppresta');
					
					$this->context->smarty->assign([
						'param_title' => $this->l('Payment cancelation done','otppresta'),
						'param_lines' => $this->filter_displayable($response_data, true),
						'param_link'  => '',
						'shop_name'   => Configuration::get('PS_SHOP_NAME')
					]);
					$order = new Order((int)$order_id);
					$customer = new Customer((int) $order->id_customer);
					
					$this->notifyCustomer(
						$customer->email, 
						$this->l('Payment cancellation','otppresta'),
						$this->context->smarty->fetch('module:otppresta/views/templates/front/success.tpl')
					);
					
				}else{
					$this->otp_error_message = $this->l('VOID failed','otppresta');
				}
				return;
			}
			
			$this->otp_error_message = $this->l('Error executing VOID transaction','otppresta');
		}catch(Exception $ex){
			$this->otp_error_message = $this->l('Error executing VOID transaction:','otppresta') . $ex->getMessage();
		}
	}
	
	public function executeCapture($order_id, $ref_transaction, $amount){
		
		$existing = $this->getTransactionData($order_id,"postauth");
		if($existing){
			if(isset($existing["processor_response_code"])){
				if($existing["processor_response_code"] == "00"){
					$this->otp_error_message = $this->l('PostAuth transaction already completed!','otppresta');
					return;
				}
			}
		}
			
		$date = new DateTime("now", new DateTimeZone("Europe/Berlin"));
		$txndatetime     = $date->format('Y:m:d-H:i:s');
		
		$chargetotal = str_replace(",",".","{$amount}");
		$chargetotal = number_format( floatval($chargetotal) , 2, '.', '');
		
		$hash = $this->generateHash(	Configuration::get("otppresta_store_sid",""), 
										$txndatetime, 
										$chargetotal, 
										$ref_transaction["currency"] , 
										Configuration::get("otppresta_shared_secret",""));
		
		
		$form_data = array(
			"txntype"     		 => "postauth",
			"timezone"    		 => "Europe/Berlin",
			"txndatetime" 		 => $txndatetime,
			"storename"   		 => Configuration::get("otppresta_store_sid",""),
			"currency"    		 => $ref_transaction["currency"],
			"oid"         		 => $ref_transaction["oid"],
			"chargetotal" 		 => $chargetotal,
			"responseSuccessURL" => $this->context->link->getModuleLink($this->name, 'readjson', array(), true),
			"responseFailURL"    => $this->context->link->getModuleLink($this->name, 'readjson', array(), true),
			"hash_algorithm"     => "SHA256",
			"hash"               => $hash
		);
		
		try{
			$resp = $this->curl_post(Configuration::get("otppresta_form_url"), $form_data);
			$response_data = array();
			$output_array = array();
			preg_match_all('/<input.*[value=|name=]\"(.*?)\".*[value=|name=]\"(.*?)\">/', $resp, $output_array);
			$names_index  = 2;
			$values_index = 1;
			
			if(in_array("response_hash",$output_array[1])){
				$names_index  = 1;
				$values_index = 2;
			}
			
			for($i = 0; $i < count($output_array[1]); $i++){
				$response_data[$output_array[$names_index][$i]] = $output_array[$values_index][$i];
			}
			
			if(isset($response_data["processor_response_code"])){
				$response_data["txntype"] = "postauth";
				$this->storeTransactionData($order_id,"postauth",$response_data);
				if($response_data["processor_response_code"] == "00"){
					$history = new OrderHistory();
					$history->id_order = $order_id;
					$history->changeIdOrderState((int)Configuration::get("otppresta_sale_completed"), (int)$order_id);
					
					$this->otp_message = $this->l('Capture/Post-autorization completed successfully','otppresta');
					
					$this->context->smarty->assign([
						'param_title' => $this->l('Payment capture/post-autorization done','otppresta'),
						'param_lines' => $this->filter_displayable($response_data, true),
						'param_link'  => '',
						'shop_name'   => Configuration::get('PS_SHOP_NAME')
					]);
					$order = new Order((int)$order_id);
					$customer = new Customer((int) $order->id_customer);
					
					$this->notifyCustomer(
						$customer->email, 
						$this->l('Payment capture/post-autorization','otppresta'),
						$this->context->smarty->fetch('module:otppresta/views/templates/front/success.tpl')
					);
					
					
				}else{
					$this->otp_error_message = $this->l('Capture/Post-autorization failed','otppresta');
				}
				return;
			}
			
			$this->otp_error_message = $this->l('Error executing Capture/Post-autorization transaction:','otppresta');
		}catch(Exception $ex){
			$this->otp_error_message = $this->l('Error executing Capture/Post-autorization transaction:','otppresta') . $ex->getMessage();
		}
	}
	
	public function executeCofVoid($order_id, $ref_transaction){
		try{
			
			$existing = $this->getTransactionData($order_id,"void");
			if($existing){
				if(isset($existing["processor_response_code"])){
					if($existing["processor_response_code"] == "00"){
						$this->otp_error_message = $this->l('VOID transaction already completed!','otppresta');
						return;
					}
				}
				if(isset($existing["ProcessorResponseCode"])){
					if($existing["ProcessorResponseCode"] == "00"){
						$this->otp_error_message = $this->l('VOID transaction already completed!','otppresta');
						return;
					}
				}
			}
			
			$rest_action = "voidTransaction";
		
			$url = trim(Configuration::get("otppresta_rest_api_url")) ."/{$rest_action}";
			$url = str_replace("//{$rest_action}","/{$rest_action}",$url);
			
			$data = array(
				"MerchantUsername" => trim(Configuration::get("otppresta_rest_api_username")),
				"OrderId"          => $ref_transaction["OrderId"],
				"TDate"            => $ref_transaction["TDate"]
			);
			$data["signature"] = md5($data["MerchantUsername"] . $data["OrderId"] . $data["TDate"] . Configuration::get("otppresta_rest_api_password"));
			$r = $this->execute_curl_request($url, $data);
			
			if($r){
				if(!isset($r->errno)){
					if(isset($r->response)){
						if(isset($r->response["ProcessorResponseCode"])){
							$r->response["txntype"] = "void";
							$this->storeTransactionData($order_id,"void",$r->response);
							if($r->response["ProcessorResponseCode"] == "00"){
								$history = new OrderHistory();
								$history->id_order = $order_id;
								$history->changeIdOrderState((int)Configuration::get("otppresta_void_completed"), (int)$order_id);
								
								$this->otp_message = $this->l('VOID completed successfully','otppresta');
								
								$this->context->smarty->assign([
									'param_title' => $this->l('Payment cancelation done','otppresta'),
									'param_lines' => $this->filter_displayable($r->response, true),
									'param_link'  => '',
									'shop_name'   => Configuration::get('PS_SHOP_NAME')
								]);
								$order = new Order((int)$order_id);
								$customer = new Customer((int) $order->id_customer);
								
								$this->notifyCustomer(
									$customer->email, 
									$this->l('Payment cancellation','otppresta'),
									$this->context->smarty->fetch('module:otppresta/views/templates/front/success.tpl')
								);
								
							}else{
								$this->otp_error_message = $this->l('VOID failed','otppresta');
							}
							
							return;
						}
					}
				}else{
					$this->otp_error_message = $this->l('Error executing VOID transaction:','otppresta') . $r->error_message;
					return;
				}
			}			
			$this->otp_error_message = $this->l('Error executing VOID transaction','otppresta');
			
		}catch(Exception $ex){
			$this->otp_error_message = $this->l('Error executing VOID transaction:','otppresta') . $ex->getMessage();
		}
	}
	
	public function executeCharge($order_id, $amount, $exchange_rate){
		try{
			
			$existing = $this->getTransactionData($order_id,"sale");
			if($existing){
				if(isset($existing["processor_response_code"])){
					if($existing["processor_response_code"] == "00"){
						$this->otp_error_message = $this->l('Sale transaction already completed!','otppresta');
						return;
					}
				}
				if(isset($existing["ProcessorResponseCode"])){
					if($existing["ProcessorResponseCode"] == "00"){
						$this->otp_error_message = $this->l('Sale transaction already completed!','otppresta');
						return;
					}
				}
			}
			
			$rest_action = "executeHostedDataTransaction";
			$url = trim(Configuration::get("otppresta_rest_api_url")) ."/{$rest_action}";
			$url = str_replace("//{$rest_action}","/{$rest_action}",$url);
			
			$chargetotal     = number_format(round(floatval($amount),2),2, '.', '');
			$date = new DateTime("now", new DateTimeZone("Europe/Berlin") );
			$txndatetime     = $date->format('Y:m:d-H:i:s');
			
			$order = new Order((int)$order_id);
			$customer = new Customer((int) $order->id_customer);
			
			
			$customer_uid    = $this->getCustomerUid($customer->email);
			
			$store_sid = Configuration::get("otppresta_store_sid");
			
			$payment_currency  = Configuration::get("otppresta_payment_currency","RSD");
			
			$otp_currencies   = $this->loadCurrencies();
		 
			$currency_num_code = $otp_currencies[$payment_currency]["currency_numeric_code"];
			
			$data = array(
				"MerchantUsername"  => Configuration::get("otppresta_rest_api_username"),
				"CustomerID"        => substr($customer->email,0,32),
				"HostedDataID"      => $customer_uid,
				"HostedDataStoreID" => $store_sid,
				"ChargeTotal"       => $chargetotal,
				"Currency"          => $currency_num_code,
				"trxType"           => 'sale',
				"metadata"          => "#" . $order_id
			);
			
			$data["signature"] = md5($data["MerchantUsername"] . $data["HostedDataID"] . $data["ChargeTotal"] . Configuration::get("otppresta_rest_api_password"));
			$r = $this->execute_curl_request($url, $data);
			
			if($r){
				if(!isset($r->errno)){
					if(isset($r->response)){
						if(isset($r->response["ProcessorResponseCode"])){
							$r->response["txntype"] = "sale";
							$this->storeTransactionData($order_id,"sale",$r->response);
							if($r->response["ProcessorResponseCode"] == "00"){
								
								$history = new OrderHistory();
								$history->id_order = $order_id;
								$history->changeIdOrderState((int)Configuration::get("otppresta_sale_completed"), (int)$order_id);
								
								$order->addOrderPayment($amount / $exchange_rate, $this->displayName ,$r->response["IpgTransactionId"]);
								
								$this->otp_message = $this->l('COF charge completed successfully','otppresta');
								
								$this->context->smarty->assign([
									'param_title' => $this->l('Card charge done','otppresta'),
									'param_lines' => $this->filter_displayable($r->response, true),
									'param_link'  => '',
									'shop_name'   => Configuration::get('PS_SHOP_NAME')
								]);
								
								$this->notifyCustomer(
									$customer->email, 
									$this->l('Your card has been charged','otppresta'),
									$this->context->smarty->fetch('module:otppresta/views/templates/front/success.tpl')
								);
								
							}else{
								$this->otp_error_message = $this->l('COF charge failed','otppresta');
							}
							
							return;
						}
					}
				}else{
					$this->otp_error_message = $this->l('Error executing COF charge transaction:','otppresta') . $r->error_message;
					return;
				}
			}		
			
			$this->otp_error_message = $this->l('Error executing COF charge transaction','otppresta');
			
			
		}catch(Exception $ex){
			$this->otp_error_message = $this->l('Error executing COF charge transaction:','otppresta') . $ex->getMessage();
		}
	}
	
    public function hookDisplayBeforePayment()
    {
	    /* Place your code here. */
    }

    public function hookDisplayHeader()
    {
        /* Place your code here. */
    }

    public function hookDisplayInvoice()
    {
        /* Place your code here. */
    }

    public function hookDisplayOrderConfirmation()
    {
        /* Place your code here. */
    }

    public function hookDisplayOrderDetail()
    {
        /* Place your code here. */
    }

    public function hookDisplayPayment()
    {
		/* Place your code here. */
    }

    public function hookDisplayPaymentReturn()
    {
        /* Place your code here. */
    }
	
	public function hookDisplayAdminOrderLeft($params){
		
		$order_id = (int) $params["id_order"];
		$order = new Order( $order_id );
		$transations = $this->getOrderTransactions($order_id);
		
		echo "<div class='otp_box panel'>";
		echo "<h4>" . $this->l('OTP options:','otppresta') . "</h4>";
		
		if(isset($this->otp_message)){
			echo "<div class='alert alert-success' role='alert'>{$this->otp_message}</div>";
		}
		
		if(isset($this->otp_error_message)){
			echo "<div class='alert alert-danger' role='alert'>{$this->otp_error_message}</div>";
		}
		
		echo "<div>";
		
		$cmd_void    = false;
		$cmd_capture = false;
		$cmd_charge  = false;
		
		if(isset($transations["preauth"])){
			if(isset($transations["preauth"]["processor_response_code"])){
				if($transations["preauth"]["processor_response_code"] == "00"){
					$cmd_capture = true;
					$cmd_void    = true;
				}
			}
		}
		
		if($cmd_capture){
			if(isset($transations["postauth"])){
				if(isset($transations["postauth"]["processor_response_code"])){
					if($transations["postauth"]["processor_response_code"] == "00"){
						$cmd_capture = false;
						$cmd_void    = true;
					}
				}
			}
		}
		
		if(!$cmd_capture){
			
			if(isset($transations["sale"])){
				if(isset($transations["sale"]["processor_response_code"])){
					if($transations["sale"]["processor_response_code"] != "00"){
						$cmd_charge = true;
					}else
						$cmd_void    = true;
				}else if(isset($transations["sale"]["ProcessorResponseCode"])){
					if($transations["sale"]["ProcessorResponseCode"] != "00"){
						$cmd_charge = true;
					}else
						$cmd_void    = true;
				}else if(Configuration::get("otppresta_merchant_type","") == "cof_capable")
					$cmd_charge = true;
			}else if(Configuration::get("otppresta_merchant_type","") == "cof_capable")
				$cmd_charge = true;
			
		}
		
		if(empty($transations)){
			$cmd_void = false;
		}
		
		if($cmd_void){
			echo "<div style='border-radius:10px;background:rgba(0,0,0,0.05);margin:5px;padding:5px;' ><h4>" . $this->l('Payment cancellation','otppresta') . "</h4>
			<form method='POST'>
			<input type='hidden' name='otp_do_transaction' value='void'/>  
			<input type='hidden' name='otp_transaction_order_id' value='{$order_id}'/> 
			<button class='btn btn-danger' id='cmd_otp_void' order_id='{$order_id}'>" . $this->l('Execute VOID','otppresta') . "</button></form></div>";
		}
		
		if($cmd_capture){
			$to_capture = number_format(round(floatval(str_ireplace(",",".",$transations["preauth"]["chargetotal"])),2),2, '.', '');
			$currency  = new Currency((int) $order->id_currency);
			$exchange_rate = $this->getExchangeRate($currency->iso_code,Configuration::get("otppresta_payment_currency","RSD"));
			
			echo "<div style='border-radius:10px;background:rgba(0,0,0,0.05);margin:5px;padding:5px;'><h4>" . $this->l('Payment completion','otppresta') . "</h4>
			<form method='POST'>
			<input type='hidden' name='otp_do_transaction' value='capture'/>  
			<input type='hidden' name='otp_transaction_order_id' value='{$order_id}'/>
			<input min='0' max='{$to_capture}'  step='0.01'  type='number' name='otp_capture_amount' id='otp_capture_amount' value='{$to_capture}' /> ". Configuration::get("otppresta_payment_currency","RSD") 
			."<br/><button style='margin-top:5px' class='btn btn-danger' id='cmd_otp_capture' order_id='{$order_id}'>" . $this->l('Execute Capture','otppresta') . "</button></form></div>";
		}
		
		if($cmd_charge){
			$to_charge = number_format(round(floatval($order->total_paid),2),2, '.', '');
			$currency  = new Currency((int) $order->id_currency);
			$exchange_rate = $this->getExchangeRate($currency->iso_code,Configuration::get("otppresta_payment_currency","RSD"));
			if($currency->iso_code != Configuration::get("otppresta_payment_currency","RSD")){
				$to_charge = floatval($to_charge) * $exchange_rate;
				$to_charge = number_format(round(floatval($to_charge),2),2, '.', '');
			}
			
			$customer = new Customer((int) $order->id_customer);
			$cuid = $this->getCustomerUid($customer->email);
			$cof_consent =	Configuration::get("otppresta_{$cuid}_cof_consent", "yes") == "yes";
			
			echo "<div style='border-radius:10px;background:rgba(0,0,0,0.05);margin:5px;padding:5px;'><h4>" . $this->l('Execute C.O.F. charge','otppresta') . "</h4>
			<form method='POST'>
			<input type='hidden' name='otp_do_transaction' value='charge'/>  
			<input type='hidden' name='otp_transaction_order_id' value='{$order_id}'/>
			<input type='hidden' name='otp_exchange_rate' value='{$exchange_rate}'/>
			<input min='0' type='number' step='0.01' name='otp_charge_amount' id='otp_charge_amount' value='{$to_charge}' />" . Configuration::get("otppresta_payment_currency","RSD") 
			. "<br/>";
			
			if($cof_consent)
				echo "<button style='margin-top:5px' class='btn btn-danger' id='cmd_otp_charge' order_id='{$order_id}'>" . $this->l('Charge','otppresta') . "</button>";
			else
				echo "<label style='color:red'>--" . $this->l('no customer C.O.F. consent','otppresta') . "--</label>";
			echo "</form></div>";
		}
		
		echo "</div>";
		echo "<h4>" . $this->l('Payment transactions log','otppresta') . "</h4>";
		
		foreach($transations as $type => $transation){
			echo "<div style='border-radius:10px;display:inline-block;background:rgba(0,0,0,0.05);margin:5px;padding:5px;'>";
			echo "<h4>" . $this->l($type) . "</h4>";
			echo "<p style='padding:1px;margin:0;font-size:11px;'>" . implode("</p><p style='padding:1px;margin:0;font-size:11px;'>",$this->filter_displayable($transation, true)) . "</p>";
			echo "</div>";
		}
		
		echo "</div>";
	}
	
}
